/***************************************************************************
 *
 * Copyright (c) 2013,2014 Codethink Limited
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#include "Log.h"
#include "WindowSystems/ButtonInputEventDispatcher.h"
#include "WindowSystems/SliderInputEventDispatcher.h"
#include "WindowSystems/TouchAreaInputEventDispatcher.h"
#include "WindowSystems/InputEventProcessor.h"
#include "WindowSystems/WaylandEvdevInputEvent.h"
#include "Subdivision.h"

using namespace LayerManagerCalibration;
using namespace InputEventProcessing;
using namespace std;


InputEventProcessor::InputEventProcessor(InputDeviceConfigurationMap* configs)
    : m_pDeviceConfigs(configs)
{
    InputEventDispatcher* dispatcher;

    dispatcher = new TouchAreaInputEventDispatcher();
    dispatcher->setListener(this);
    m_dispatchers.insert( pair<Subdivision::Type, InputEventDispatcher*>
                              (Subdivision::TOUCH, dispatcher) );

    dispatcher = new ButtonInputEventDispatcher();
    dispatcher->setListener(this);
    m_dispatchers.insert( pair<Subdivision::Type, InputEventDispatcher*>
                               (Subdivision::BUTTON, dispatcher) );

    dispatcher = new SliderInputEventDispatcher();
    dispatcher->setListener(this);
    m_dispatchers.insert( pair<Subdivision::Type, InputEventDispatcher*>
                               (Subdivision::SLIDER, dispatcher) );

}

InputEventProcessor::~InputEventProcessor()
{
    InputEventDispatcherMap::iterator iter;
    for (iter = m_dispatchers.begin();
         iter != m_dispatchers.end();
         iter++)
    {
        delete iter->second;
    }
}

bool InputEventProcessor::deviceNeedsProcessing(const string& deviceName) const
{
    if (m_pDeviceConfigs->find(deviceName) != m_pDeviceConfigs->end())
    {
        return true;
    }
    else
    {
        return false;
    }
}

void InputEventProcessor::processButtonEvent(struct evdev_input_device *device,
                                             uint32_t time,
                                             uint32_t button,
                                             wl_pointer_button_state state)
{
    LOG_DEBUG("InputEventProcessor",
              "device name=" << device->deviceName << ", "
              "time=" << time << ", "
              "button=" << button << ", state=" << state);

    struct TouchEvent event = {
        {
            (uint) device->abs.x, // coord.x
            (uint) device->abs.y // coord.y
        },
        INPUT_STATE_OTHER, // state
        (uint) device->st_slot, // slot
        button, // button
        time, // time
        true // isSingleTouch
    };

    switch (state)
    {
    case WL_POINTER_BUTTON_STATE_PRESSED:
        event.state = INPUT_STATE_PRESSED;
        break;
    case WL_POINTER_BUTTON_STATE_RELEASED:
        event.state = INPUT_STATE_RELEASED;
        break;
    default:
        LOG_WARNING("InputEventProcessor",
                    "Unknown value of state=" << state);
        return;
    }

    processTouchEvent(device, event);
}

void InputEventProcessor::processTouchEvent(struct evdev_input_device *device,
                                            struct TouchEvent& event)
{
    // obtain the configuration for the input device
    InputDeviceConfiguration* config = m_pDeviceConfigs->find(device->deviceName)->second;

    // obtain the configuration for the device and look up the subdivision
    // corresponding to the raw input coordinates
    // This will also convert the raw coordinate to a calibrated coordinate
    // if the subdivision is calibrated
    coordinate rawCoordinate = {
        event.coord.x,
        event.coord.y,
    };

    Subdivision* subdivision;
    coordinate calibratedCoordinate;

    config->getSubdivisionForCoordinate(rawCoordinate, &subdivision,
                                        calibratedCoordinate);

    // TODO: clip coordinate to device boundaries
    coordinate clippedCoordinate = {
        calibratedCoordinate.x,
        calibratedCoordinate.y,
    };


    // obtain the active subdivision for the device
    Subdivision* activeSubdivision = m_state.getActiveSubdivision(device, event.slot);

    LOG_DEBUG("InputEventProcessor",
              "device name=" << device->deviceName << ", "
              "coordinate x=" << event.coord.x << ", y=" << event.coord.y);

    if (subdivision != activeSubdivision)
    {
        // active subdivision and new subdivision differ, leave the
        // active subdivision
        if (activeSubdivision != NULL)
        {
            coordinate activeCalibratedCoordinate;

            if (activeSubdivision->getUncalibrated())
            {
                activeCalibratedCoordinate.x = rawCoordinate.x;
                activeCalibratedCoordinate.y = rawCoordinate.y;
            }
            else
            {
                config->getCalibration().transformCoordinate(rawCoordinate, activeCalibratedCoordinate);
            }

            InputEventDispatcher* dispatcher = m_dispatchers.find(activeSubdivision->getType())->second;
            dispatcher->processLeave(activeSubdivision, device, activeCalibratedCoordinate, event);
        }

        // set active subdivision to the new subdivision
        m_state.setActiveSubdivision(device, event.slot, subdivision);

        // enter the new subdivision
        if (subdivision != NULL)
        {
            InputEventDispatcher* dispatcher = m_dispatchers.find(subdivision->getType())->second;
            dispatcher->processEnter(subdivision, device, clippedCoordinate, event);
        }
    }

    // handle the touch event for this subdivision in the
    // correct dispatcher
    if (subdivision != NULL)
    {
        InputEventDispatcher* dispatcher = m_dispatchers.find(subdivision->getType())->second;
        dispatcher->processTouchEvent(subdivision, device, clippedCoordinate, event);
    }
}

void InputEventProcessor::processMotionEvent(struct evdev_input_device *device,
                                             uint32_t time)
{
    struct TouchEvent event = {
        {
            (uint) device->abs.x, // coord.x
            (uint) device->abs.y // coord.y
        },
        INPUT_STATE_MOTION, // state
        (uint) device->st_slot, // slot
        0, // button
        time, // time
        true // isSingleTouch
    };

    processTouchEvent(device, event);
}

void InputEventProcessor::processMultiTouchEvent(struct evdev_input_device* device,
                                                 uint32_t time)
{
    struct TouchEvent event = {
        {
            (uint) device->mt.x[device->mt.slot], // coord.x
            (uint) device->mt.y[device->mt.slot] // coord.y
        },
        INPUT_STATE_OTHER, // state
        (uint) device->mt.slot, // slot
        0, // button
        time, // time
        false // isSingleTouch
    };

    if (device->pending_events & EVDEV_ABSOLUTE_MT_DOWN)
    {
        event.state = INPUT_STATE_PRESSED;
        processTouchEvent(device, event);
    }
    if (device->pending_events & EVDEV_ABSOLUTE_MT_MOTION)
    {
        event.state = INPUT_STATE_MOTION;
        processTouchEvent(device, event);
    }
    if (device->pending_events & EVDEV_ABSOLUTE_MT_UP)
    {
        event.state = INPUT_STATE_RELEASED;
        processTouchEvent(device, event);
    }
}

void InputEventProcessor::generateKeyEvent(struct evdev_input_device* device,
                                           uint time,
                                           uint code,
                                           wl_keyboard_key_state state,
                                           bool treatPressAsRelease)
{
    LOG_DEBUG("InputEventProcessor",
              "device name=" << device->deviceName << ", "
              "time=" << time << ", "
              "code=" << code << ", "
              "state=" << state);

    WaylandEvdevInputEvent *inputEvent = static_cast<WaylandEvdevInputEvent*>(device->master);
    if (inputEvent == NULL)
    {
        LOG_WARNING("InputEventProcessor",
                    "Failed to generate key event for device, "
                    "name=" << device->deviceName);
    }
    else
    {
        inputEvent->notifyKey(device, time, code, state, true, treatPressAsRelease);
    }
}


void InputEventProcessor::generateTouchEvent(struct evdev_input_device* device,
                                             uint eventType,
                                             coordinate& coord)
{
    LOG_DEBUG("InputEventProcessor",
              "device name=" << device->deviceName << ", "
              "type=" << eventType << ", "
              "coordinate x=" << coord.x << ", y=" << coord.y);

    WaylandEvdevInputEvent *inputEvent = static_cast<WaylandEvdevInputEvent*>(device->master);
    if (inputEvent == NULL)
    {
        LOG_WARNING("InputEventProcessor",
                    "Failed to get input event for device, "
                    "name=" << device->deviceName);
    }
    else
    {
        WLEvent wlEvent;
        InputEventState eventState = INPUT_STATE_OTHER;

        if (eventType == EVDEV_ABSOLUTE_MT_DOWN)
        {
            wlEvent.x = coord.x;
            wlEvent.y = coord.y;
            wlEvent.touchId = device->mt.slot;
            wlEvent.touchType = WL_TOUCH_DOWN;
            eventState = INPUT_STATE_PRESSED;
        }
        else if (eventType == EVDEV_ABSOLUTE_MT_UP)
        {
            wlEvent.x = coord.x;
            wlEvent.y = coord.y;
            wlEvent.touchId = device->mt.slot;
            wlEvent.touchType = WL_TOUCH_UP;
            eventState = INPUT_STATE_RELEASED;
        }

        if (eventState != INPUT_STATE_OTHER)
        {
            device->pWlInputEvent->windowSystem().manageWLInputEvent(INPUT_DEVICE_TOUCH,
                                                          eventState,
                                                          &wlEvent);
        }
        else
        {
            LOG_WARNING("InputEventProcessor",
                        "Failed to generate event of unknown "
                        "type=" << eventType << ", "
                        "device name=" << device->deviceName);
        }
    }
}


void InputEventProcessor::generateMotionEvent(struct evdev_input_device* device,
                                              uint eventType,
                                              coordinate& coord)
{
    LOG_DEBUG("InputEventProcessor",
              "device name=" << device->deviceName << ", "
              "type=" << eventType << ", "
              "coordinate x=" << coord.x << ", y=" << coord.y);

    WaylandEvdevInputEvent *inputEvent = static_cast<WaylandEvdevInputEvent*>(device->master);
    if (inputEvent == NULL)
    {
        LOG_WARNING("InputEventProcessor",
                    "Failed to get input event for device, "
                    "name=" << device->deviceName);
    }
    else
    {
        WLEvent wlEvent;
        InputEventState eventState = INPUT_STATE_OTHER;
        unsigned int deviceType = 0;

        if (eventType == EVDEV_ABSOLUTE_MT_MOTION)
        {
            deviceType = INPUT_DEVICE_TOUCH;
            eventState = INPUT_STATE_MOTION;
            wlEvent.x = coord.x;
            wlEvent.y = coord.y;
            wlEvent.touchId = device->mt.slot;
            wlEvent.touchType = WL_TOUCH_MOTION;
        }
        else if (eventType == EVDEV_ABSOLUTE_MOTION)
        {
            deviceType = INPUT_DEVICE_POINTER;
            eventState = INPUT_STATE_MOTION;
            wlEvent.x = coord.x;
            wlEvent.y = coord.y;

            struct wl_seat *wlSeat = device->pWlInputEvent->inputDevice().seat();
            wlSeat->pointer->x = wl_fixed_from_int(coord.x);
            wlSeat->pointer->y = wl_fixed_from_int(coord.y);
        }

        if (eventState != INPUT_STATE_OTHER)
        {
            device->pWlInputEvent->windowSystem().manageWLInputEvent(deviceType,
                                                          eventState,
                                                          &wlEvent);
        }
        else
        {
            LOG_WARNING("InputEventProcessor",
                        "Failed to generate event of "
                        "unknown type=" << eventType);
        }
    }
}

void InputEventProcessor::generateButtonEvent(struct evdev_input_device* device,
                                              uint time,
                                              int posx,
                                              int posy,
                                              uint id,
                                              enum wl_pointer_button_state state)
{

    LOG_DEBUG("InputEventProcessor",
              "device name=" << device->deviceName << ", "
              "time=" << time << ", "
              "id=" << id << ", "
              "state=" << state << ", "
              "coordinate x=" << posx << ", y=" << posy);

    WaylandEvdevInputEvent *inputEvent = static_cast<WaylandEvdevInputEvent*>(device->master);
    if (!inputEvent)
    {
        LOG_WARNING("InputEventProcessor",
                    "Failed to get input event for device, "
                    "name=" << device->deviceName);
        return;
    }

    struct wl_seat* wlSeat = device->pWlInputEvent->inputDevice().seat();
    wlSeat->pointer->x = wl_fixed_from_int(posx);
    wlSeat->pointer->y = wl_fixed_from_int(posy);

    inputEvent->notifyButton(device, time, id, state);
}

void InputEventProcessor::generateAxisEvent(struct evdev_input_device* device,
                                            uint time,
                                            uint id,
                                            float value)
{
    LOG_DEBUG("InputEventProcessor",
              "device name=" << device->deviceName << ", "
              "time=" << time << ", "
              "id=" << id << ", "
              "value=" << value);

    WaylandEvdevInputEvent *inputEvent = static_cast<WaylandEvdevInputEvent*>(device->master);
    if (inputEvent == NULL)
    {
        LOG_WARNING("InputEventProcessor",
                    "Failed to get input event for device, "
                    "name=" << device->deviceName);
    }
    else
    {
        WLEvent wlEvent;
        wlEvent.axis = id;
        wlEvent.axisValue = wl_fixed_from_double(value);

        device->pWlInputEvent->windowSystem().manageWLInputEvent(INPUT_DEVICE_POINTER,
                                                      INPUT_STATE_AXIS,
                                                      &wlEvent);
    }
}
